--[[
	SowingMachineRollerReady.lua
	
	Autor: 		Ifko[nator]
	Datum: 		12.11.2025
	Version: 	2.3
	
	Changelog:	v1.0 @24.12.2021 - initial implementation in FS 22
				---------------------------------------------------------------------------------------------------------------
				v1.1 @22.04.2022 - fix for patch 1.4 and higher
				---------------------------------------------------------------------------------------------------------------
				v1.2 @18.07.2022 - fix for sowing machines with fertilizer
				---------------------------------------------------------------------------------------------------------------
				v2.0 @29.06.2025 - convert to FS 25
				---------------------------------------------------------------------------------------------------------------
				v2.1 @15.09.2025 - fix for sowing machines with fertilizing function and precision farming
				---------------------------------------------------------------------------------------------------------------
				v2.2 @04.11.2025 - fix for Patch 1.14.0.0
				---------------------------------------------------------------------------------------------------------------
				v2.3 @12.11.2025 - fix for manure barrels with slurry cultivator attached in combination with precision farming
]]

SowingMachineRollerReady = {};
SowingMachineRollerReady.currentModName = "";
SowingMachineRollerReady.precisionFarmingModTable = {};

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "SOWING MACHINE ROLLER READY" or mod.title:upper() == "WALZFUNKTION FÜR SÄMASCHINEN" then		
		if g_modIsLoaded[mod.modName] then	
			SowingMachineRollerReady.currentModName = mod.modName;
			
			break;
		end;
	end;
end;

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "PRECISION FARMING" then		
		if g_modIsLoaded[mod.modName] then	
			SowingMachineRollerReady.precisionFarmingModTable = _G[mod.modName];
			
			break;
		end;
	end;
end;

function SowingMachineRollerReady.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).sowingMachineRollerReady#allowRollerSowingMachine", "Allow/Disallow roller function.", false);
end;

function SowingMachineRollerReady.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(SowingMachine, specializations);
end;

function SowingMachineRollerReady.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdateTick",
		"onReadStream",
		"onWriteStream",
		"onRegisterActionEvents"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, SowingMachineRollerReady);
	end;
end;

function SowingMachineRollerReady.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "setAllowRollerSowingMachine", SowingMachineRollerReady.setAllowRollerSowingMachine);
end;

function SowingMachineRollerReady:onLoad(savegame)
	local specSowingMachine = self.spec_sowingMachine;
	
	specSowingMachine.isRollerAllowed = self.xmlFile:getValue("vehicle.sowingMachine.fieldGroundType#value") == "SOWN";
	specSowingMachine.allowRollerSowingMachine = false;

	if not specSowingMachine.isRollerAllowed then
		return;
	end;

	if savegame ~= nil and not savegame.resetVehicles then
		specSowingMachine.allowRollerSowingMachine = savegame.xmlFile:getValue(savegame.key .. ".sowingMachineRollerReady#allowRollerSowingMachine", specSowingMachine.allowRollerSowingMachine);
	end;

	self:setAllowRollerSowingMachine(specSowingMachine.allowRollerSowingMachine, false);

	specSowingMachine.l10nTexts = {};

	local l10nTexts = {
		"action_activateRoller",
		"action_deactivateRoller"
	};

	for _, l10nText in pairs(l10nTexts) do
		specSowingMachine.l10nTexts[l10nText] = g_i18n:getText(l10nText, SowingMachineRollerReady.currentModName);
	end;
end;

function SowingMachineRollerReady:onReadStream(streamId, connection)
	self.spec_sowingMachine.allowRollerSowingMachine = streamReadBool(streamId);
end;

function SowingMachineRollerReady:onWriteStream(streamId, connection)
	streamWriteBool(streamId, self.spec_sowingMachine.allowRollerSowingMachine);
end;

function SowingMachineRollerReady:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
		local specSowingMachine = self.spec_sowingMachine;

		if isActiveForInputIgnoreSelection and specSowingMachine.isRollerAllowed then
			local _, actionEventId = self:addActionEvent(specSowingMachine.actionEvents, InputAction.SET_ALLOW_ROLLER_BUTTON, self, SowingMachineRollerReady.actionEventSetAllowRollerSowingMachine, false, true, false, true, nil);
			
			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH);
			g_inputBinding:setActionEventTextVisibility(actionEventId, true);
			g_inputBinding:setActionEventActive(actionEventId, true);
			g_inputBinding:setActionEventText(actionEventId, specSowingMachine.l10nTexts.action_activateRoller);
		end;
	end;
end;

function SowingMachineRollerReady:onUpdateTick(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local specSowingMachine = self.spec_sowingMachine;

	if self:getFillUnitFillLevel(specSowingMachine.fillUnitIndex) <= 0 and self:getFillUnitCapacity(specSowingMachine.fillUnitIndex) ~= 0 then
		if self:getIsTurnedOn() then
			g_currentMission:showBlinkingWarning(g_i18n:getText("warning_pleaseFillFirstSowingMachine", SowingMachineRollerReady.currentModName), 5000);

			self:setIsTurnedOn(false);
		end;
	end;

	if not specSowingMachine.isRollerAllowed then
		return;
	end;

	local actionEvent = specSowingMachine.actionEvents[InputAction.SET_ALLOW_ROLLER_BUTTON];

	if actionEvent ~= nil then
		local actionEventText = specSowingMachine.l10nTexts.action_activateRoller;

		if specSowingMachine.allowRollerSowingMachine then
			actionEventText = specSowingMachine.l10nTexts.action_deactivateRoller;
		end;

		g_inputBinding:setActionEventText(actionEvent.actionEventId, actionEventText);
	end;
end;

function SowingMachineRollerReady.actionEventSetAllowRollerSowingMachine(self, actionName, inputValue, callbackState, isAnalog)
	self:setAllowRollerSowingMachine(not self.spec_sowingMachine.allowRollerSowingMachine, false);
end;

function SowingMachineRollerReady:setAllowRollerSowingMachine(allowRollerSowingMachine, noEventSend)
	self.spec_sowingMachine.allowRollerSowingMachine = allowRollerSowingMachine;
	
	SowingMachineRollerReadyEvent.sendEvent(self, allowRollerSowingMachine, noEventSend);
end;

function SowingMachineRollerReady:saveToXMLFile(xmlFile, key)
	local specSowingMachine = self.spec_sowingMachine;
	
	if specSowingMachine.allowRollerSowingMachine then	
		xmlFile:setValue(key .. "#allowRollerSowingMachine", specSowingMachine.allowRollerSowingMachine);
	end;
end;

function SowingMachineRollerReady:processSowingMachineArea(superFunc, workArea, dt)
	local changedArea, totalArea = superFunc(self, workArea, dt);

	local specSowingMachine = self.spec_sowingMachine;
	
	if specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		changedArea, _ = changedArea + FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
		
		specSowingMachine.workAreaParameters.lastChangedArea = specSowingMachine.workAreaParameters.lastChangedArea + changedArea;
		specSowingMachine.workAreaParameters.lastStatsArea = specSowingMachine.workAreaParameters.lastStatsArea + changedArea;
	end;

	return changedArea, totalArea;
end;

SowingMachine.processSowingMachineArea = Utils.overwrittenFunction(SowingMachine.processSowingMachineArea, SowingMachineRollerReady.processSowingMachineArea);

function SowingMachineRollerReady:processFertilizingSowingMachineArea(superFunc, superFuncFertilizingSowingMachine, workArea, dt)
	local specSowingMachine = self.spec_sowingMachine;
    local specSprayer = self.spec_sprayer;

    local sprayerParams = specSprayer.workAreaParameters;
    local sowingParams = specSowingMachine.workAreaParameters;

    local changedArea, totalArea;

	local rootVehicle = self.rootVehicle;

    specSowingMachine.isWorking = self:getLastSpeed() > 0.5;

    if specSowingMachine.waterSeeding and not self.isInWater then
        specSowingMachine.showWaterPlantingRequiredWarning = true;

        if self:getIsAIActive() then
            rootVehicle:stopCurrentAIJob(AIMessageErrorNoFieldFound.new());
        end;

        return 0, 0;
    elseif not specSowingMachine.waterSeeding and self.isInWater then
        specSowingMachine.showWaterPlantingProhibitedWarning = true;

        if self:getIsAIActive() then
           rootVehicle:stopCurrentAIJob(AIMessageErrorNoFieldFound.new());
        end;

        return 0, 0;
    end;

    if not sowingParams.isActive then
        return 0, 0;
    end;

    if not self:getIsAIActive() or not g_currentMission.missionInfo.helperBuySeeds then
        if sowingParams.seedsVehicle == nil then
            if self:getIsAIActive() then
                rootVehicle:stopCurrentAIJob(AIMessageErrorOutOfFill.new());
            end;

            return 0, 0;
        end;
    end;

    --## stop the sowing machine if the fertilizer tank was filled and got empty
    --## do not stop if the tank was all the time empty
    if (specSprayer.isSlurryTanker and g_currentMission.missionInfo.helperSlurrySource == 1)
    or (specSprayer.isManureSpreader and g_currentMission.missionInfo.helperManureSource == 1)
    or (specSprayer.isFertilizerSprayer and not g_currentMission.missionInfo.helperBuyFertilizer) then
        if self:getIsAIActive() then
            if sprayerParams.sprayFillType == nil or sprayerParams.sprayFillType == FillType.UNKNOWN then
                if sprayerParams.lastAIHasSprayed ~= nil then
                    rootVehicle:stopCurrentAIJob(AIMessageErrorOutOfFill.new());

                    sprayerParams.lastAIHasSprayed = nil;
                end
            else
                sprayerParams.lastAIHasSprayed = true;
            end;
        end;
    end;

    if not sowingParams.canFruitBePlanted then
        return 0, 0;
    end;

    local startX, _, startZ = getWorldTranslation(workArea.start);
    local widthX, _, widthZ = getWorldTranslation(workArea.width);
    local heightX, _, heightZ = getWorldTranslation(workArea.height);

    --## remove tireTracks
    FSDensityMapUtil.eraseTireTrack(startX, startZ, widthX, widthZ, heightX, heightZ);

    if not self.isServer and self.currentUpdateDistance > FertilizingSowingMachine.CLIENT_DM_UPDATE_RADIUS then
        return 0, 0;
    end;

    --## we need to use fertilizer as spraying type because fertilizer is the final blocking value
    local sprayTypeIndex = SprayType.FERTILIZER;

    if sprayerParams.sprayFillLevel <= 0 or (self.spec_fertilizingSowingMachine.needsSetIsTurnedOn and not self:getIsTurnedOn()) then
        sprayTypeIndex = nil;
    end;

    local fruitTypeDesc = g_fruitTypeManager:getFruitTypeByIndex(specSowingMachine.workAreaParameters.seedsFruitType);

    if fruitTypeDesc.seedRequiredFieldType ~= nil then
        local cx, cz = (startX + widthX + heightX) / 3, (startZ + widthZ + heightZ) / 3;

        if FSDensityMapUtil.getFieldTypeAtWorldPos(cx, cz) ~= fruitTypeDesc.seedRequiredFieldType then
            if fruitTypeDesc.seedRequiredFieldType == FieldType.RICE then
                specSowingMachine.showFieldTypeWarningRiceRequired = true;
            else
                specSowingMachine.showFieldTypeWarningRegularRequired = true;
            end;

            if self:getIsAIActive() then
                rootVehicle:stopCurrentAIJob(AIMessageErrorNoFieldFound.new());
            end;
        end;
    end;

    if not specSowingMachine.useDirectPlanting then
        changedArea, totalArea = FSDensityMapUtil.updateSowingArea(sowingParams.seedsFruitType, startX, startZ, widthX, widthZ, heightX, heightZ, sowingParams.fieldGroundType, sowingParams.ridgeSeeding, sowingParams.angle, nil, sprayTypeIndex);
    else
        changedArea, totalArea = FSDensityMapUtil.updateDirectSowingArea(sowingParams.seedsFruitType, startX, startZ, widthX, widthZ, heightX, heightZ, sowingParams.fieldGroundType, sowingParams.ridgeSeeding, sowingParams.angle, nil, sprayTypeIndex);
    end;

    specSowingMachine.isProcessing = specSowingMachine.isWorking;

    if sprayTypeIndex ~= nil then
        local sprayAmount = specSprayer.doubledAmountIsActive and 2 or 1;
        local sprayChangedArea, sprayTotalArea = FSDensityMapUtil.updateSprayArea(startX, startZ, widthX, widthZ, heightX, heightZ, sprayTypeIndex, sprayAmount);

        sprayerParams.lastChangedArea = sprayerParams.lastChangedArea + sprayChangedArea;
        sprayerParams.lastTotalArea = sprayerParams.lastTotalArea + sprayTotalArea;
        sprayerParams.lastStatsArea = 0;
        sprayerParams.isActive = true;
        sprayerParams.lastSprayTime = g_time;

        local farmId = self:getLastTouchedFarmlandFarmId();
        local ha = MathUtil.areaToHa(sprayerParams.lastChangedArea, g_currentMission:getFruitPixelsToSqm());

        g_farmManager:updateFarmStats(farmId, "sprayedHectares", ha);
        g_farmManager:updateFarmStats(farmId, "sprayedTime", dt / (1000 * 60));
        g_farmManager:updateFarmStats(farmId, "sprayUsage", sprayerParams.usage);
    end;

    sowingParams.lastChangedArea = sowingParams.lastChangedArea + changedArea;
    sowingParams.lastStatsArea = sowingParams.lastStatsArea + changedArea;
    sowingParams.lastTotalArea = sowingParams.lastTotalArea + totalArea;

    self:updateMissionSowingWarning(startX, startZ);

	if specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		changedArea, _ = changedArea + FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
		
		specSowingMachine.workAreaParameters.lastChangedArea = specSowingMachine.workAreaParameters.lastChangedArea + changedArea;
		specSowingMachine.workAreaParameters.lastStatsArea = specSowingMachine.workAreaParameters.lastStatsArea + changedArea;
	end;

	return changedArea, totalArea;
end;

function SowingMachineRollerReady:preProcessExtUnderRootFertilizerArea(superFunc, workArea, dt)
    superFunc(self, workArea, dt);

	local specSowingMachine = self.spec_sowingMachine;

	if specSowingMachine ~= nil and specSowingMachine.isRollerAllowed and specSowingMachine.allowRollerSowingMachine then
		local sx, _ ,sz = getWorldTranslation(workArea.start);
		local wx, _ ,wz = getWorldTranslation(workArea.width);
		local hx, _ ,hz = getWorldTranslation(workArea.height);

		FSDensityMapUtil.updateRollerArea(sx, sz, wx, wz, hx, hz, specSowingMachine.workAreaParameters.angle);
	end;
end

if SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer == nil then
	FertilizingSowingMachine.processSowingMachineArea = Utils.overwrittenFunction(FertilizingSowingMachine.processSowingMachineArea, SowingMachineRollerReady.processFertilizingSowingMachineArea);
else
	SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer.preProcessExtUnderRootFertilizerArea = Utils.overwrittenFunction(SowingMachineRollerReady.precisionFarmingModTable.ExtendedSprayer.preProcessExtUnderRootFertilizerArea, SowingMachineRollerReady.preProcessExtUnderRootFertilizerArea);
end;