--[[
Copyright (C) GtX (Andy), 2025

Author: GtX | Andy
Date: 01.05.2025
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

MoveAnimalsEvent = {}

MoveAnimalsEvent.SUCCESS = 0
MoveAnimalsEvent.NO_ACCESS = 1
MoveAnimalsEvent.OBJECT_DOES_NOT_EXIST = 2
MoveAnimalsEvent.INVALID_CLUSTER = 3
MoveAnimalsEvent.ANIMAL_NOT_SUPPORTED = 4
MoveAnimalsEvent.NOT_ENOUGH_ANIMALS = 5
MoveAnimalsEvent.NOT_ENOUGH_SPACE = 6
MoveAnimalsEvent.NOT_ENOUGH_MONEY = 7

local MoveAnimalsEvent_mt = Class(MoveAnimalsEvent, Event)
InitEventClass(MoveAnimalsEvent, "MoveAnimalsEvent")

function MoveAnimalsEvent.emptyNew()
    return Event.new(MoveAnimalsEvent_mt)
end

function MoveAnimalsEvent.new(sourceObject, targetObject, clusterId, numAnimals, transportCost)
    local self = MoveAnimalsEvent.emptyNew()

    self.sourceObject = sourceObject
    self.targetObject = targetObject

    self.clusterId = clusterId
    self.numAnimals = numAnimals

    self.transportCost = transportCost

    return self
end

function MoveAnimalsEvent.newServerToClient(errorCode)
    local self = MoveAnimalsEvent.emptyNew()

    self.errorCode = errorCode

    return self
end

function MoveAnimalsEvent:readStream(streamId, connection)
    if not connection:getIsServer() then
        self.sourceObject = NetworkUtil.readNodeObject(streamId)
        self.targetObject = NetworkUtil.readNodeObject(streamId)

        self.clusterId = streamReadInt32(streamId)
        self.numAnimals = streamReadUInt8(streamId)

        self.transportCost = streamReadInt32(streamId)
    else
        self.errorCode = streamReadUIntN(streamId, 3)
    end

    self:run(connection)
end

function MoveAnimalsEvent:writeStream(streamId, connection)
    if connection:getIsServer() then
        NetworkUtil.writeNodeObject(streamId, self.sourceObject)
        NetworkUtil.writeNodeObject(streamId, self.targetObject)

        streamWriteInt32(streamId, self.clusterId)
        streamWriteUInt8(streamId, self.numAnimals)

        streamWriteInt32(streamId, math.abs(self.transportCost))
    else
        streamWriteUIntN(streamId, self.errorCode, 3)
    end
end

function MoveAnimalsEvent:run(connection)
    if not connection:getIsServer() then
        local errorCode = MoveAnimalsEvent.NO_ACCESS

        if g_currentMission:getHasPlayerPermission("tradeAnimals", connection) then
            local uniqueUserId = g_currentMission.userManager:getUniqueUserIdByConnection(connection)
            local farm = g_farmManager:getFarmForUniqueUserId(uniqueUserId)

            errorCode = MoveAnimalsEvent.validate(self.sourceObject, self.targetObject, self.clusterId, self.numAnimals, self.transportCost, farm.farmId)

            if errorCode == MoveAnimalsEvent.SUCCESS then
                local cluster = self.sourceObject:getClusterById(self.clusterId)
                local newCluster = cluster:clone()

                newCluster:changeNumAnimals(self.numAnimals)
                self.targetObject:addCluster(newCluster)

                local clusterSystem = self.sourceObject:getClusterSystem()

                cluster:changeNumAnimals(-self.numAnimals)
                clusterSystem:updateNow()

                if self.transportCost > 0 then
                    g_currentMission:addMoney(-self.transportCost, farm.farmId, MoneyType.OTHER, true, true)
                end
            end
        end

        connection:sendEvent(MoveAnimalsEvent.newServerToClient(errorCode))
    else
        g_messageCenter:publish(MoveAnimalsEvent, self.errorCode)
    end
end

function MoveAnimalsEvent.validate(sourceObject, targetObject, clusterId, numAnimals, transportCost, farmId)
    if sourceObject == nil or targetObject == nil then
        return MoveAnimalsEvent.OBJECT_DOES_NOT_EXIST
    end

    if not g_currentMission.accessHandler:canFarmAccess(farmId, sourceObject) or not g_currentMission.accessHandler:canFarmAccess(farmId, targetObject) then
        return MoveAnimalsEvent.NO_ACCESS
    end

    local cluster = sourceObject:getClusterById(clusterId)

    if cluster == nil then
        return MoveAnimalsEvent.INVALID_CLUSTER
    end

    if not targetObject:getSupportsAnimalSubType(cluster:getSubTypeIndex()) then
        return MoveAnimalsEvent.ANIMAL_NOT_SUPPORTED
    end

    if cluster:getNumAnimals() < numAnimals then
        return MoveAnimalsEvent.NOT_ENOUGH_ANIMALS
    end

    if targetObject:getNumOfFreeAnimalSlots(cluster:getSubTypeIndex()) < numAnimals then
        return MoveAnimalsEvent.NOT_ENOUGH_SPACE
    end

    transportCost = math.abs(transportCost or 0)

    if transportCost > 0 and g_currentMission:getMoney(farmId) - transportCost < 0 then
        return MoveAnimalsEvent.NOT_ENOUGH_MONEY
    end

    return MoveAnimalsEvent.SUCCESS
end

function MoveAnimalsEvent.getErrorCodeText(errorCode)
    if errorCode ~= nil then
        if errorCode == MoveAnimalsEvent.SUCCESS then
            return g_i18n:getText("shop_movedToFarm"), true
        end

        if errorCode == MoveAnimalsEvent.NO_ACCESS then
            return g_i18n:getText("shop_messageNoPermissionToTradeAnimals"), false
        end

        if errorCode == MoveAnimalsEvent.OBJECT_DOES_NOT_EXIST then
            return g_i18n:getText("shop_messageHusbandryDoesNotExist"), false
        end

        if errorCode == MoveAnimalsEvent.INVALID_CLUSTER then
            return g_i18n:getText("shop_messageInvalidCluster"), false
        end

        if errorCode == MoveAnimalsEvent.ANIMAL_NOT_SUPPORTED then
            return g_i18n:getText("shop_messageAnimalTypeNotSupported"), false
        end

        if errorCode == MoveAnimalsEvent.NOT_ENOUGH_ANIMALS then
            return g_i18n:getText("shop_messageNotEnoughAnimals"), false
        end

        if errorCode == MoveAnimalsEvent.NOT_ENOUGH_SPACE then
            return g_i18n:getText("shop_messageNotEnoughSpaceAnimals"), false
        end

        if errorCode == MoveAnimalsEvent.NOT_ENOUGH_MONEY then
            return g_i18n:getText("shop_messageNotEnoughMoneyToBuy"), false
        end
    end

    return "", true
end
